package edu.uky.ai.lp.logic;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

/**
 * A unifier tracks which variable are equal to one another and which variable
 * are equal to constants.
 * 
 * @author Stephen G. Ware
 */
public class Unifier implements Cloneable {

	/**
	 * Represents a group of variables with the game value.
	 * 
	 * @author Stephen G. Ware
	 */
	private final class Group {
		
		/** The variables */
		public final Variable[] variables;
		
		/** The value (false if not set) */
		public final Constant value;
		
		/**
		 * Constructs a new group.
		 * 
		 * @param variables the variables
		 * @param value the value
		 */
		private Group(Variable[] variables, Constant value) {
			this.variables = variables;
			this.value = value;
		}
		
		/**
		 * Constructs a new group.
		 * 
		 * @param variable the variable
		 */
		public Group(Variable variable) {
			this(new Variable[]{ variable }, null);
		}
		
		@Override
		public String toString() {
			if(variables.length == 1 && value == null)
				return variables[0] + "/?";
			String str = "";
			for(Variable variable : variables)
				str += variable + "/";
			if(value == null)
				return str.substring(0, str.length() - 1);
			else
				return str + value;
		}
		
		/**
		 * Returns a new group the same as this one, but with its value set if
		 * possible.
		 * 
		 * @param value the value to set
		 * @return a group with value set, or null if this group already has a different value
		 */
		public Group set(Constant value) {
			if(this.value == null)
				return new Group(variables, value);
			else if(this.value.equals(value))
				return this;
			else
				return null;
		}
		
		/**
		 * Merges two groups.
		 * 
		 * @param other the other group to merge
		 * @return the resulting group, or null if the groups cannot be merged
		 */
		public Group merge(Group other) {
			if(value == null || other.value == null || value.equals(other.value)) {
				Variable[] variables = Arrays.copyOf(this.variables, this.variables.length + other.variables.length);
				System.arraycopy(other.variables, 0, variables, this.variables.length, other.variables.length);
				return new Group(variables, other.value == null ? this.value : other.value);
			}
			else
				return null;
		}
	}
	
	/** A hash table tracking which variable belong to which groups */
	private final HashMap<Variable, Group> groups;
	
	@SuppressWarnings("unchecked")
	private Unifier(Unifier toClone) {
		this.groups = (HashMap<Variable, Group>) toClone.groups.clone();
	}
	
	/**
	 * Constructs an empty unifier.
	 */
	public Unifier() {
		this.groups = new HashMap<>();
	}
	
	@Override
	public String toString() {
		Set<Group> groups = new HashSet<>();
		for(Group group : this.groups.values())
			groups.add(group);
		String str = "";
		for(Group group : groups)
			str += ", " + group;
		if(str.length() > 0)
			str = str.substring(1);
		return "{" + str + " }";
	}
	
	/**
	 * Returns the group associated with a given variable, constructing a new
	 * one if necessary.
	 * 
	 * @param variable the variable
	 * @return its group
	 */
	private final Group getGroup(Variable variable) {
		Group group = groups.get(variable);
		if(group == null)
			group = new Group(variable);
		return group;
	}
	
	/**
	 * Sets all variables in a group equal to the group in the hash table.
	 * 
	 * @param group the group to set
	 */
	private final void setGroup(Group group) {
		for(Variable variable : group.variables)
			groups.put(variable, group);
	}
	
	@Override
	public Unifier clone() {
		return new Unifier(this);
	}
	
	/**
	 * Returns the value assigned to a variable (if any) under this unifier.
	 * 
	 * @param variable the variable
	 * @return the value assigned to the variable (if any)
	 */
	public Term get(Variable variable) {
		Group group = getGroup(variable);
		if(group.value == null)
			return group.variables[0];
		else
			return group.value;
	}
	
	/**
	 * Sets a variable equal to a value (if value is a constant) or as having
	 * the same value as another variable (if value is a variable).  Note that
	 * this object is not modified; a new unifier is returned.
	 * 
	 * @param variable the variable to set
	 * @param value the value to set it to
	 * @return the resulting unifier, or null if the variable cannot be set to that value
	 */
	public Unifier set(Variable variable, Term value) {
		Group group = getGroup(variable);
		if(value instanceof Variable)
			group = group.merge(getGroup((Variable) value));
		else
			group = group.set((Constant) value);
		if(group == null)
			return null;
		Unifier clone = clone();
		clone.setGroup(group);
		return clone;
	}
}
